/*!
*******************************************************************************
* \file             spi_tclMySpinAudioStream.cpp
* \brief            Implements the Audio functionality for mySPIN
*******************************************************************************
\verbatim
PROJECT:        Gen3 Projects
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Audio Streaming (Lib) Implementation for mySPIN using gstreamer
COPYRIGHT:      &copy; BSOT

HISTORY:
Date       |  Author                          | Modifications
19.02.2016 |  J. Schmidt                      | initial version

\endverbatim
******************************************************************************/

// FIXME: known bug: device type AUDIO_DEVICE_TYPE_ALSA needs mute/unmute on HU for audio to be heard



//*****************************************************************************/
/* INCLUDES                                                                   */
/******************************************************************************/
#include "mspin_audio_stream.h"
#include "spi_tclMySpinAudioStream.h"
#include "mspin_logging.h"


/***************************************************************************
** FUNCTION:  S32 spi_tclMySpinAudioStream::check_start_audio()
***************************************************************************/
/*!
* \fn      check_start_audio()
* \brief   check, whether audio down streaming can be started; if yes, start
*
***************************************************************************/
S32 spi_tclMySpinAudioStream::check_start_audio()
{

    if ( (NULL == m_poGstreamer) || (NULL == m_poSndCtrlMonitor) )
    {
        return -1;            // error
    }else{

        // check, whether input & output device have been configured and
        // start was commanded
        if ( m_szInDeviceName.empty() || m_szOutDeviceName.empty() || ((AUDIO_DEVICE_TYPE_BT != m_enAudioStreamDeviceType) && (!m_configReceived)) || !m_startReceived)
        {
            return 1;
        }


        if (AUDIO_DEVICE_TYPE_ALSA_W_SNDCTL == m_enAudioStreamDeviceType)
        {
            m_poSndCtrlMonitor->vInitMonitor(this);

            // if called from vSampleRateChanged_CB then m_rateReceived == true and bGetInitCaptureEnabled() will not be called
            if (!m_rateReceived && !m_poSndCtrlMonitor->bGetInitCaptureEnabled()) {
                return 1;             // can't start yet
            }else{
                // case: m_rateReceived || m_poSndCtrlMonitor->bGetInitCaptureEnabled()
                m_poGstreamer->mspin_audio_start(m_szInDeviceName.c_str(), m_szOutDeviceName.c_str(), m_enAudioStreamDeviceType);
                return 0;            // OK
            }
        }else{
            m_poGstreamer->mspin_audio_start(m_szInDeviceName.c_str(), m_szOutDeviceName.c_str(), m_enAudioStreamDeviceType);
            return 0;                // OK
        }
    }

}


/***************************************************************************
** FUNCTION:  S32 spi_tclMySpinAudioStream::SetAudioStreamConfig()
***************************************************************************/
/*!
* \fn      SetAudioStreamConfig()
* \brief   configure stream parameters (ignored for BT audio)
*
***************************************************************************/
S32 spi_tclMySpinAudioStream::SetAudioStreamConfig(const audioStreamConfig corAudioStreamConfig)
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s()(%d) ENTER",__PRETTY_FUNCTION__, corAudioStreamConfig.sampleRate);

    if ( NULL != m_poGstreamer )
    {
        m_poGstreamer->mspin_audio_set_config(corAudioStreamConfig);
        m_configReceived = true;

        return check_start_audio();
    }
    else
    {
        return -1;
    }
}


/***************************************************************************
** FUNCTION:  S32 spi_tclMySpinAudioStream::SetAudioStreamInputDevice()
***************************************************************************/
/*!
* \fn      SetAudioStreamInputDevice()
* \brief   set ALSA name of input device
*
***************************************************************************/
S32 spi_tclMySpinAudioStream::SetAudioStreamInputDevice( std::string szInDeviceName, audioStreamDeviceType enAudioStreamDeviceType )
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s()(%s) ENTER",__PRETTY_FUNCTION__,szInDeviceName.c_str() );

    m_szInDeviceName = szInDeviceName;
    m_enAudioStreamDeviceType = enAudioStreamDeviceType;

    return check_start_audio();
}


/***************************************************************************
** FUNCTION:  S32 spi_tclMySpinAudioStream::SetAudioStreamOutputDevice()
***************************************************************************/
/*!
* \fn      SetAudioStreamInputDevice()
* \brief   set ALSA name of output device
*
***************************************************************************/
S32 spi_tclMySpinAudioStream::SetAudioStreamOutputDevice( std::string szOutDeviceName )
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s()(%s) ENTER",__PRETTY_FUNCTION__,szOutDeviceName.c_str() );

    m_szOutDeviceName = szOutDeviceName;

    return check_start_audio();
}


/***************************************************************************
** FUNCTION:  S32 spi_tclMySpinAudioStream::StartAudioStream()
***************************************************************************/
/*!
* \fn      StartAudioStream()
* \brief   starts audio streaming for normal audio (down streaming)
*
***************************************************************************/
S32 spi_tclMySpinAudioStream::StartAudioStream()
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s() ENTER",__PRETTY_FUNCTION__ );

    m_startReceived = true;

    return check_start_audio();
}


/***************************************************************************
** FUNCTION:  S32 spi_tclMySpinAudioStream::StopAudioStream()
***************************************************************************/
/*!
* \fn      StopAudioStream()
* \brief   stops audio streaming for normal audio (down streaming)
*
***************************************************************************/
S32 spi_tclMySpinAudioStream::StopAudioStream()
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s() ENTER",__PRETTY_FUNCTION__ );

    // clear input & output device names
    m_szOutDeviceName = "";
    m_szInDeviceName = "";
    m_startReceived = false;
    m_configReceived = false;
    m_rateReceived = false;

    if ( (NULL != m_poGstreamer) && (NULL != m_poSndCtrlMonitor) )
    {
        if (AUDIO_DEVICE_TYPE_ALSA_W_SNDCTL == m_enAudioStreamDeviceType)
        {
            m_poSndCtrlMonitor->vStopMonitor();
        }

        m_poGstreamer->mspin_audio_stop();
    }

    return 0;
}

/***************************************************************************
** FUNCTION:  S32 spi_tclMySpinAudioStream::vRegisterCallbacks()
***************************************************************************/
/*!
* \fn      vRegisterCallbacks()
* \brief   register callback functions for error and sample rate change
*
***************************************************************************/
void spi_tclMySpinAudioStream::vRegisterCallbacks( U32 cou32Context , audioStreamCallbacks rAudioStreamCallbacks )
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s() ENTER",__PRETTY_FUNCTION__ );

    if( NULL != m_poGstreamer )
    {
        m_poGstreamer->mspin_audio_register_callbacks(cou32Context , rAudioStreamCallbacks);
    }
}


/***************************************************************************
** FUNCTION:  S32 spi_tclMySpinAudioStream::vSampleRateChanged_CB()
***************************************************************************/
/*!
* \fn      vSampleRateChanged_CB()
* \brief   on sample rate change call handler registered to spi_tclMySpinAudioStream
*            this function is the registered callback of SndCtrlMonitor
*
***************************************************************************/
void spi_tclMySpinAudioStream::vSampleRateChanged_CB(U32 u32SampleRate,bool bEndPointEnabled)
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s(%d, %d) ENTER",__PRETTY_FUNCTION__, u32SampleRate, bEndPointEnabled );

    if ( NULL != m_poGstreamer )
    {
        if (bEndPointEnabled) {
            m_poGstreamer->mspin_audio_set_rate(u32SampleRate);

            if (!m_rateReceived) {
                mspin_log_printLn(eMspinVerbosityInfo, "%s() start audio",__PRETTY_FUNCTION__ );
                m_rateReceived = true;
                check_start_audio();                    // first time: start audio
            }else{
                if (u32SampleRate != m_lastRate) {      // restart audio

                    mspin_log_printLn(eMspinVerbosityInfo, "%s() restart audio",__PRETTY_FUNCTION__ );
                    m_poGstreamer->mspin_audio_stop();
                    check_start_audio();
                }
            }
            m_lastRate = u32SampleRate;

            // call the registered handler (only if bEndPointEnabled)
            m_poGstreamer->mspin_audio_handle_rate_changed(u32SampleRate);

        }else{
            mspin_log_printLn(eMspinVerbosityInfo, "%s() stop audio",__PRETTY_FUNCTION__ );
            m_poGstreamer->mspin_audio_stop();            // stop audio
            m_rateReceived = false;
        }
    }
}


/***************************************************************************
** FUNCTION:  spi_tclMySpinAudioStream::spi_tclMySpinAudioStream()
***************************************************************************/
spi_tclMySpinAudioStream::spi_tclMySpinAudioStream()
    : m_poGstreamer(NULL)
    , m_poSndCtrlMonitor(NULL)
    , m_szInDeviceName("")
    , m_szOutDeviceName("")
    , m_enAudioStreamDeviceType(AUDIO_DEVICE_TYPE_BT)
    , m_startReceived(false)
    , m_configReceived(false)
    , m_rateReceived(false)
    , m_lastRate(0)
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s() ENTER",__PRETTY_FUNCTION__ );

    m_poGstreamer = new spi_tclMySpinAudioGstreamer();
    m_poSndCtrlMonitor = new spi_tclMySPINSndCtrlMonitor();
}


/***************************************************************************
** FUNCTION:  spi_tclMySpinAudioStream::~spi_tclMySpinAudioStream()
***************************************************************************/
spi_tclMySpinAudioStream::~spi_tclMySpinAudioStream()
{
    mspin_log_printLn(eMspinVerbosityInfo, "%s() ENTER",__PRETTY_FUNCTION__ );

    if( NULL != m_poGstreamer )
    {
        delete m_poGstreamer;
        m_poGstreamer = NULL;
    }

    if( NULL != m_poSndCtrlMonitor )
    {
        delete m_poSndCtrlMonitor;
        m_poSndCtrlMonitor = NULL;
    }
}

